Rem FindChangeByList.vbs
Rem An InDesign VBScript
Rem
Rem Loads a series of tab-delimited strings from a text file, then performs a series
Rem of find/change operations based on the strings read from the file.
Rem
Rem The data file is tab-delimited, with carriage returns separating records.
Rem
Rem The format of each record in the file is:
Rem findType<tab>findProperties<tab>changeProperties<tab>findChangeOptions<tab>description
Rem 
Rem Where:
Rem <tab> is a tab character
Rem findType is "text", "grep", or "glyph" (this sets the type of find/change operation to use).
Rem findProperties is a properties record (as text) of the find preferences.
Rem changeProperties is a properties record (as text) of the change preferences.
Rem findChangeOptions is a properties record (as text) of the find/change options.
Rem description is a description of the find/change operation
Rem 
Rem Very simple example:
Rem text	{findWhat:"--"}	{changeTo:"^_"}	{includeFootnotes:true, includeMasterPages:true, includeHiddenLayers:true, wholeWord:false}	Find all double dashes and replace with an em dash.
Rem 
Rem More complex example:
Rem text	{findWhat:"^9^9.^9^9"}	{appliedCharacterStyle:"price"}	{include footnotes:true, include master pages:true, include hidden layers:true, whole word:false}	Find $10.00 to $99.99 and apply the character style "price".
Rem 
Rem All InDesign search metacharacters are allowed in the "find what" and "change to" properties.
Rem
Rem For more on InDesign/InCopy scripting see the documentation included in the Scripting SDK 
Rem available at http://www.adobe.com/devnet/indesign/sdk.html
Rem or visit the InDesign Scripting User to User forum at http://www.adobeforums.com
Rem
main
Function main()
	Set myInDesign = CreateObject("InDesign.Application.2022")
	Rem Set the user interaction level to allow the display of dialog boxes and alerts.
	myInDesign.ScriptPreferences.UserInteractionLevel = idUserInteractionLevels.idInteractWithAll
	If myInDesign.Documents.Count > 0 Then
	    If myInDesign.Documents.Item(1).Stories.Count > 0 Then
	        If myInDesign.Selection.Count > 0 Then
	            Select Case TypeName(myInDesign.Selection.Item(1))
	                Case "InsertionPoint", "Character", "Word", "TextStyleRange", "Line", "Paragraph", "TextColumn", "Text", "TextFrame", "Cell", "Table", "Row", "Column"
	                    myDisplayDialog(myInDesign)
	                Case Else:
	                    myFindChangeByList myInDesign, myInDesign.Documents.Item(1)
	            End Select
	     	Else
	            myFindChangeByList myInDesign, myInDesign.Documents.Item(1)
	        End If
	    Else
	        MsgBox ("The current document contains no text. Please open a document containing text and try again.")
	    End If
	Else
	    MsgBox ("No documents are open. Please open a document and try again.")
	End If
End Function
Function myDisplayDialog(myInDesign)
    myObject = "None"
    Set myDialog = myInDesign.Dialogs.Add
    myDialog.Name = "FindChangeByList"
    With myDialog.DialogColumns.Add
        With .DialogRows.Add
            With .DialogColumns.Add
                With .StaticTexts.Add
                    .StaticLabel = "Search Range"
                End With
            End With
            With .DialogColumns.Add
                Set myRangeButtons = .RadiobuttonGroups.Add
                With myRangeButtons
                    With .RadiobuttonControls.Add
                        .StaticLabel = "Document"
                        .CheckedState = True
                    End With
                    With .RadiobuttonControls.Add
                        .StaticLabel = "Selected Story"
                    End With
                    If myInDesign.Selection.Item(1).Contents <> "" Then
                        With .RadiobuttonControls.Add
                            .StaticLabel = "Selection"
	                        .CheckedState = True
                        End With
                    End If
                End With
            End With
        End With
    End With
    myResult = myDialog.Show
    If myResult = True Then
        Select Case myRangeButtons.SelectedButton
            Case 0:
                Set myObject = myInDesign.Documents.Item(1)
            Case 1:
                Select Case TypeName(myInDesign.Selection.Item(1))
                    Case "InsertionPoint", "Character", "Word", "TextStyleRange", "Line", "Paragraph", "TextColumn", "Text", "TextFrame"
                        Set myObject = myInDesign.Selection.Item(1).ParentStory
                End Select
            Case 2:
                Select Case TypeName(myInDesign.Selection.Item(1))
                    Case "Character", "Word", "TextStyleRange", "Line", "Paragraph", "TextColumn", "Text", "TextFrame"
                        If myInDesign.Selection.Item(1).Contents <> "" Then
                            Set myObject = myInDesign.Selection.Item(1).Texts.Item(1)
						Else
							Set myObject = myInDesign.Selection.Item(1).ParentStory
						End If
                Case Else:
                    Set myObject = Nothing
            End Select
        End Select
		myFindChangeByList myInDesign, myObject
    Else
        myDialog.Destroy
    End If
End Function
Function myFindChangeByList(myInDesign, myObject)
    myFileName = myFindFile(myInDesign, "\FindChangeSupport\FindChangeList.txt")
    If myFileName <> "" Then
        Set myFileSystemObject = CreateObject("Scripting.FileSystemObject")
        Set myFindChangeFile = myFileSystemObject.OpenTextFile(myFileName)
        Rem Loop through the find/change operations.
        Do While Not myFindChangeFile.AtEndOfStream
            myLine = myFindChangeFile.ReadLine
            Rem Ignore empty lines.
            If (Left(myLine, 4) = "text")Or(Left(myLine, 4) = "grep")Or(Left(myLine, 5) = "glyph") Then
                myFindChangeArray = Split(myLine, vbTab)
                Rem The first field in the line is the FindType string.
				myFindType = myFindChangeArray(0)
                Rem The second field in the line is the FindPreferences string.
                myFindPreferences = myFindChangeArray(1)
                Rem The third field in the line is the ChangePreferences string.
                myChangePreferences = myFindChangeArray(2)
                Rem The fourth field in the line can be either "once" or "all".
                Rem If it's "all", keep on searching until no instances of the
                Rem search text are found, if it's "once," search just once.
                myFindChangeOptions = myFindChangeArray(3)
                Select Case myFindType
                	Case "text"
                		myFindText myInDesign, myObject, myFindPreferences, myChangePreferences, myFindChangeOptions
                	Case "grep"
                		myFindGrep myInDesign, myObject, myFindPreferences, myChangePreferences, myFindChangeOptions
                	Case "glyph"
                		myFindGlyph myInDesign, myObject, myFindPreferences, myChangePreferences, myFindChangeOptions
                End Select
            End If
        Loop
        myFindChangeFile.Close
    End If
End Function
Function myFindText(myInDesign, myObject, myFindPreferences, myChangePreferences, myFindChangeOptions)
	Rem Clear the FindText/ChangeText Preferences before each find/change operation.
	myInDesign.FindTextPreferences = idNothingEnum.idNothing
	myInDesign.ChangeTextPreferences = idNothingEnum.idNothing
	myScript = "app.findTextPreferences.properties = " & myFindPreferences & ";" & vbCr
	myScript = myScript & "app.changeTextPreferences.properties = " & myChangePreferences & ";" & vbCr
	myScript = myScript & "app.findChangeTextOptions.properties = " & myFindChangeOptions & ";" & vbCr
    myInDesign.DoScript myScript, idScriptLanguage.idJavascript
	Set myFoundItems = myObject.ChangeText
	Rem Clear the FindText/ChangeText Preferences after each find/change operation.
	myInDesign.FindTextPreferences = idNothingEnum.idNothing
	myInDesign.ChangeTextPreferences = idNothingEnum.idNothing
End Function
Function myFindGrep(myInDesign, myObject, myFindPreferences, myChangePreferences, myFindChangeOptions)
	Rem Clear the FindGrep/ChangeGrep Preferences before each find/change operation.
	myInDesign.FindGrepPreferences = idNothingEnum.idNothing
	myInDesign.ChangeGrepPreferences = idNothingEnum.idNothing
	myScript = "app.findGrepPreferences.properties = " & myFindPreferences & ";" & vbCr
	myScript = myScript & "app.changeGrepPreferences.properties = " & myChangePreferences & ";" & vbCr
	myScript = myScript & "app.findChangeGrepOptions.properties = " & myFindChangeOptions & ";" & vbCr
    myInDesign.DoScript myScript, idScriptLanguage.idJavascript
	Set myFoundItems = myObject.ChangeGrep
	Rem Clear the FindGrep/ChangeGrep Preferences after each find/change operation.
	myInDesign.FindGrepPreferences = idNothingEnum.idNothing
	myInDesign.ChangeGrepPreferences = idNothingEnum.idNothing
End Function
Function myFindGlyph(myInDesign, myObject, myFindPreferences, myChangePreferences, myFindChangeOptions)
	Rem Clear the FindGlyph/ChangeGlyph Preferences before each find/change operation.
	myInDesign.FindGlyphPreferences = idNothingEnum.idNothing
	myInDesign.ChangeGlyphPreferences = idNothingEnum.idNothing
	myScript = "app.findGlyphPreferences.properties = " & myFindPreferences & ";" & vbCr
	myScript = myScript & "app.changeGlyphPreferences.properties = " & myChangePreferences & ";" & vbCr
	myScript = myScript & "app.findChangeGlyphOptions.properties = " & myFindChangeOptions & ";" & vbCr
    myInDesign.DoScript myScript, idScriptLanguage.idJavascript
	Set myFoundItems = myObject.ChangeGlyph
	Rem Clear the FindGlyph/ChangeGlyph Preferences after each find/change operation.
	myInDesign.FindGlyphPreferences = idNothingEnum.idNothing
	myInDesign.ChangeGlyphPreferences = idNothingEnum.idNothing
End Function 
Function myFindFile(myInDesign, myRelativePath)
	myFilePath = myInDesign.ActiveScript
	Set myFileSystemObject = CreateObject("Scripting.FileSystemObject")
	myFolderPath = myFileSystemObject.GetFile(myFilePath).ParentFolder.Path
	myFilePath = myFolderPath & myRelativePath
	If myFileSystemObject.FileExists(myFilePath) = True Then
		myFile = myFilePath
	Else
		myFile = myGetFileName(myFolderPath)
	End If
	myFindFile = myFile
End Function
Function myGetFileName(myFolder)
	Set myDialog = CreateObject("UserAccounts.CommonDialog")
	myDialog.Filter = "VBScript files|*.vbs"
	myDialog.InitialDir = myFolder
	myResult = myDialog.ShowOpen	 
	If myResult = 0 Then
	    myFileName = ""
	Else
		myFileName = myDialog.FileName
	End If
	myGetFileName = myFileName
End Function
'' SIG '' Begin signature block
'' SIG '' MIIhPQYJKoZIhvcNAQcCoIIhLjCCISoCAQExDzANBglg
'' SIG '' hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
'' SIG '' BgEEAYI3AgEeMCQCAQEEEE7wKRaZJ7VNj+Ws4Q8X66sC
'' SIG '' AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
'' SIG '' qwVL5GR4Il5JDbCeOITBzQs8hpNefWOdIMoqURhrqiig
'' SIG '' ggyUMIIF0DCCBLigAwIBAgIQCHK5i9aO5k82w+FiDw5C
'' SIG '' dzANBgkqhkiG9w0BAQsFADBsMQswCQYDVQQGEwJVUzEV
'' SIG '' MBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3
'' SIG '' d3cuZGlnaWNlcnQuY29tMSswKQYDVQQDEyJEaWdpQ2Vy
'' SIG '' dCBFViBDb2RlIFNpZ25pbmcgQ0EgKFNIQTIpMB4XDTIw
'' SIG '' MTIxOTAwMDAwMFoXDTIyMTIyMTIzNTk1OVowgeYxEzAR
'' SIG '' BgsrBgEEAYI3PAIBAxMCVVMxGTAXBgsrBgEEAYI3PAIB
'' SIG '' AhMIRGVsYXdhcmUxHTAbBgNVBA8MFFByaXZhdGUgT3Jn
'' SIG '' YW5pemF0aW9uMRAwDgYDVQQFEwcyNzQ4MTI5MQswCQYD
'' SIG '' VQQGEwJVUzELMAkGA1UECBMCY2ExETAPBgNVBAcTCFNh
'' SIG '' biBKb3NlMRMwEQYDVQQKEwpBZG9iZSBJbmMuMSwwKgYD
'' SIG '' VQQLEyNJbGx1c3RyYXRvciwgSW5EZXNpZ24sIEluQ29w
'' SIG '' eSwgTXVzZTETMBEGA1UEAxMKQWRvYmUgSW5jLjCCASIw
'' SIG '' DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKraHbKp
'' SIG '' 7LWhQb1eUijAwzznOPnL0ar6Bh5IyTX0X3Nmx0y/zOPp
'' SIG '' A769ixYCLdufIms+U9xBOpXey6jajunMtomUtqLHkYMN
'' SIG '' DT2jzWOCqLvnwrHjehFxLYwYIt9ymU0VqJHPcIO+0lfA
'' SIG '' s4ZoskzD5OepJsCVjURfwILDS/l51c42dKXqbPmr25vz
'' SIG '' UQBlKIg7JV7JJGj29DQMOeCI9LSY4dAUYsfaEfDx0dcb
'' SIG '' T/oDlUZAH/mGrvFonoqGwLDNNSpWkFauC3ryUlgMJ6if
'' SIG '' JX9iZIrdm6Glo/EvxbsHMzsL2TclgBXoa0mYq4PYgMH1
'' SIG '' 5XYNA9y4x7QNLRmWPpXU+4gnfN0CAwEAAaOCAfEwggHt
'' SIG '' MB8GA1UdIwQYMBaAFI/ofvBtMmoABSPHcJdqOpD/a+rU
'' SIG '' MB0GA1UdDgQWBBS5nvze3WzR/65rffm/COInbJUTtzAu
'' SIG '' BgNVHREEJzAloCMGCCsGAQUFBwgDoBcwFQwTVVMtREVM
'' SIG '' QVdBUkUtMjc0ODEyOTAOBgNVHQ8BAf8EBAMCB4AwEwYD
'' SIG '' VR0lBAwwCgYIKwYBBQUHAwMwewYDVR0fBHQwcjA3oDWg
'' SIG '' M4YxaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0VWQ29k
'' SIG '' ZVNpZ25pbmdTSEEyLWcxLmNybDA3oDWgM4YxaHR0cDov
'' SIG '' L2NybDQuZGlnaWNlcnQuY29tL0VWQ29kZVNpZ25pbmdT
'' SIG '' SEEyLWcxLmNybDBLBgNVHSAERDBCMDcGCWCGSAGG/WwD
'' SIG '' AjAqMCgGCCsGAQUFBwIBFhxodHRwczovL3d3dy5kaWdp
'' SIG '' Y2VydC5jb20vQ1BTMAcGBWeBDAEDMH4GCCsGAQUFBwEB
'' SIG '' BHIwcDAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGln
'' SIG '' aWNlcnQuY29tMEgGCCsGAQUFBzAChjxodHRwOi8vY2Fj
'' SIG '' ZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRFVkNvZGVT
'' SIG '' aWduaW5nQ0EtU0hBMi5jcnQwDAYDVR0TAQH/BAIwADAN
'' SIG '' BgkqhkiG9w0BAQsFAAOCAQEALl6DDJXB+ruS0J/n4zd6
'' SIG '' IxqKGcJnnn72tgm50zJjxjqD9f4gMlY5lFA4Dcz9D6ur
'' SIG '' Yc734LqeT9p+oUv2PI4KdFMpSmhdrKLkI8eTK46NHVW8
'' SIG '' U/GZctdsVZR1JvtI1ByrH/DkQS7HTKWwtbyOkKU7lfYK
'' SIG '' HIS+qUso64cM6M2asUym8CdN/3FbwJ9dGJDX9yfMISbX
'' SIG '' oIDfInSnVDD+Z9EnnpsEw9obZwNHaGdd1NJRuPrlwatk
'' SIG '' YdBJuo5kN5/eBMYb95FbK8zZf8e+5FTnEJ0PmxRakjlF
'' SIG '' xh5Zg3CCjiIjr3w/uUkOaa2PjpkNy/vuUYxCuJb53Ca6
'' SIG '' i+n/WMWWjYJLNDCCBrwwggWkoAMCAQICEAPxtOFfOoLx
'' SIG '' FJZ4s9fYR1wwDQYJKoZIhvcNAQELBQAwbDELMAkGA1UE
'' SIG '' BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcG
'' SIG '' A1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTErMCkGA1UEAxMi
'' SIG '' RGlnaUNlcnQgSGlnaCBBc3N1cmFuY2UgRVYgUm9vdCBD
'' SIG '' QTAeFw0xMjA0MTgxMjAwMDBaFw0yNzA0MTgxMjAwMDBa
'' SIG '' MGwxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2Vy
'' SIG '' dCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20x
'' SIG '' KzApBgNVBAMTIkRpZ2lDZXJ0IEVWIENvZGUgU2lnbmlu
'' SIG '' ZyBDQSAoU0hBMikwggEiMA0GCSqGSIb3DQEBAQUAA4IB
'' SIG '' DwAwggEKAoIBAQCnU/oPsrUT8WTPhID8roA10bbXx6Ms
'' SIG '' rBosrPGErDo1EjqSkbpX5MTJ8y+oSDy31m7clyK6UXlh
'' SIG '' r0MvDbebtEkxrkRYPqShlqeHTyN+w2xlJJBVPqHKI3zF
'' SIG '' QunEemJFm33eY3TLnmMl+ISamq1FT659H8gTy3WbyeHh
'' SIG '' ivgLDJj0yj7QRap6HqVYkzY0visuKzFYZrQyEJ+d8FKh
'' SIG '' 7+g+03byQFrc+mo9G0utdrCMXO42uoPqMKhM3vELKlhB
'' SIG '' iK4AiasD0RaCICJ2615UOBJi4dJwJNvtH3DSZAmALeK2
'' SIG '' nc4f8rsh82zb2LMZe4pQn+/sNgpcmrdK0wigOXn93b89
'' SIG '' OgklAgMBAAGjggNYMIIDVDASBgNVHRMBAf8ECDAGAQH/
'' SIG '' AgEAMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAKBggr
'' SIG '' BgEFBQcDAzB/BggrBgEFBQcBAQRzMHEwJAYIKwYBBQUH
'' SIG '' MAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBJBggr
'' SIG '' BgEFBQcwAoY9aHR0cDovL2NhY2VydHMuZGlnaWNlcnQu
'' SIG '' Y29tL0RpZ2lDZXJ0SGlnaEFzc3VyYW5jZUVWUm9vdENB
'' SIG '' LmNydDCBjwYDVR0fBIGHMIGEMECgPqA8hjpodHRwOi8v
'' SIG '' Y3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRIaWdoQXNz
'' SIG '' dXJhbmNlRVZSb290Q0EuY3JsMECgPqA8hjpodHRwOi8v
'' SIG '' Y3JsNC5kaWdpY2VydC5jb20vRGlnaUNlcnRIaWdoQXNz
'' SIG '' dXJhbmNlRVZSb290Q0EuY3JsMIIBxAYDVR0gBIIBuzCC
'' SIG '' AbcwggGzBglghkgBhv1sAwIwggGkMDoGCCsGAQUFBwIB
'' SIG '' Fi5odHRwOi8vd3d3LmRpZ2ljZXJ0LmNvbS9zc2wtY3Bz
'' SIG '' LXJlcG9zaXRvcnkuaHRtMIIBZAYIKwYBBQUHAgIwggFW
'' SIG '' HoIBUgBBAG4AeQAgAHUAcwBlACAAbwBmACAAdABoAGkA
'' SIG '' cwAgAEMAZQByAHQAaQBmAGkAYwBhAHQAZQAgAGMAbwBu
'' SIG '' AHMAdABpAHQAdQB0AGUAcwAgAGEAYwBjAGUAcAB0AGEA
'' SIG '' bgBjAGUAIABvAGYAIAB0AGgAZQAgAEQAaQBnAGkAQwBl
'' SIG '' AHIAdAAgAEMAUAAvAEMAUABTACAAYQBuAGQAIAB0AGgA
'' SIG '' ZQAgAFIAZQBsAHkAaQBuAGcAIABQAGEAcgB0AHkAIABB
'' SIG '' AGcAcgBlAGUAbQBlAG4AdAAgAHcAaABpAGMAaAAgAGwA
'' SIG '' aQBtAGkAdAAgAGwAaQBhAGIAaQBsAGkAdAB5ACAAYQBu
'' SIG '' AGQAIABhAHIAZQAgAGkAbgBjAG8AcgBwAG8AcgBhAHQA
'' SIG '' ZQBkACAAaABlAHIAZQBpAG4AIABiAHkAIAByAGUAZgBl
'' SIG '' AHIAZQBuAGMAZQAuMB0GA1UdDgQWBBSP6H7wbTJqAAUj
'' SIG '' x3CXajqQ/2vq1DAfBgNVHSMEGDAWgBSxPsNpA/i/RwHU
'' SIG '' mCYaCALvY2QrwzANBgkqhkiG9w0BAQsFAAOCAQEAGTNK
'' SIG '' DIEzN9utNsnkyTq7tRsueqLi9ENCF56/TqFN4bHb6YHd
'' SIG '' nwHy5IjV6f4J/SHB7F2A0vDWwUPC/ncr2/nXkTPObNWy
'' SIG '' GTvmLtbJk0+IQI7N4fV+8Q/GWVZy6OtqQb0c1UbVfEnK
'' SIG '' ZjgVwb/gkXB3h9zJjTHJDCmiM+2N4ofNiY0/G//V4BqX
'' SIG '' i3zabfuoxrI6Zmt7AbPN2KY07BIBq5VYpcRTV6hg5ucC
'' SIG '' EqC5I2SiTbt8gSVkIb7P7kIYQ5e7pTcGr03/JqVNYUvs
'' SIG '' RkG4Zc64eZ4IlguBjIo7j8eZjKMqbphtXmHGlreKuWEt
'' SIG '' k7jrDgRD1/X+pvBi1JlqpcHB8GSUgDGCFAEwghP9AgEB
'' SIG '' MIGAMGwxCzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdp
'' SIG '' Q2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5j
'' SIG '' b20xKzApBgNVBAMTIkRpZ2lDZXJ0IEVWIENvZGUgU2ln
'' SIG '' bmluZyBDQSAoU0hBMikCEAhyuYvWjuZPNsPhYg8OQncw
'' SIG '' DQYJYIZIAWUDBAIBBQCggZwwGQYJKoZIhvcNAQkDMQwG
'' SIG '' CisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEOMAwGCisG
'' SIG '' AQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIEWxt+hoKu2k
'' SIG '' JhNKF/UR9Bxf+tsQZd7G8r0EOFmutUnnMDAGCisGAQQB
'' SIG '' gjcCAQwxIjAgoB6AHABBAGQAbwBiAGUAIABJAG4ARABl
'' SIG '' AHMAaQBnAG4wDQYJKoZIhvcNAQEBBQAEggEAmQ5R6KAB
'' SIG '' UiNvgf1Kqr88skYO0ibpwZdmbwhkL1hEQsfEcEpNJntQ
'' SIG '' 79m+uA9QJrIZffrYm0Tx64YyW+WiPjbKya4hXqmefgDz
'' SIG '' /xopiHexX+C4JOa/r0YHpBCpkDxp66RNHTXXDG3G2kTC
'' SIG '' ZH81rpPjYcIciU7sk8xNoMngGSGTXFSnxTN8gEXmt9DE
'' SIG '' 8YqOuGBTVoSukf4pRC/0Fj+Lz2ic9Tg7wPMAd5jl/pS+
'' SIG '' eTQG+VE1h6QP7UKreH5S2qprs4OW6h5QJRWvIJ4THU05
'' SIG '' M+hUgGaT979UdBWeDb2TXxlaH7Ncc/BRQn16Q0mMewRb
'' SIG '' XWKGwNZcEepwegMiN7MlxRxzRKGCEbIwghGuBgorBgEE
'' SIG '' AYI3AwMBMYIRnjCCEZoGCSqGSIb3DQEHAqCCEYswghGH
'' SIG '' AgEDMQ8wDQYJYIZIAWUDBAIBBQAwdwYLKoZIhvcNAQkQ
'' SIG '' AQSgaARmMGQCAQEGCWCGSAGG/WwHATAxMA0GCWCGSAFl
'' SIG '' AwQCAQUABCCCQTtRNdqCUPPw1nRWFu34mqbiztv7ZIli
'' SIG '' XkCUyuA5VAIQBlVRnEo0tQ4FFqIa1xisFRgPMjAyMjA0
'' SIG '' MTgxOTAzMzNaoIINfDCCBsYwggSuoAMCAQICEAp6Soie
'' SIG '' yZlCkAZjOE2Gl50wDQYJKoZIhvcNAQELBQAwYzELMAkG
'' SIG '' A1UEBhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMu
'' SIG '' MTswOQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0IFJT
'' SIG '' QTQwOTYgU0hBMjU2IFRpbWVTdGFtcGluZyBDQTAeFw0y
'' SIG '' MjAzMjkwMDAwMDBaFw0zMzAzMTQyMzU5NTlaMEwxCzAJ
'' SIG '' BgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5j
'' SIG '' LjEkMCIGA1UEAxMbRGlnaUNlcnQgVGltZXN0YW1wIDIw
'' SIG '' MjIgLSAyMIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIIC
'' SIG '' CgKCAgEAuSqWI6ZcvF/WSfAVghj0M+7MXGzj4CUu0jHk
'' SIG '' PECu+6vE43hdflw26vUljUOjges4Y/k8iGnePNIwUQ0x
'' SIG '' B7pGbumjS0joiUF/DbLW+YTxmD4LvwqEEnFsoWImAdPO
'' SIG '' w2z9rDt+3Cocqb0wxhbY2rzrsvGD0Z/NCcW5QWpFQiNB
'' SIG '' Wvhg02UsPn5evZan8Pyx9PQoz0J5HzvHkwdoaOVENFJf
'' SIG '' D1De1FksRHTAMkcZW+KYLo/Qyj//xmfPPJOVToTpdhiY
'' SIG '' mREUxSsMoDPbTSSF6IKU4S8D7n+FAsmG4dUYFLcERfPg
'' SIG '' OL2ivXpxmOwV5/0u7NKbAIqsHY07gGj+0FmYJs7g7a5/
'' SIG '' KC7CnuALS8gI0TK7g/ojPNn/0oy790Mj3+fDWgVifnAs
'' SIG '' 5SuyPWPqyK6BIGtDich+X7Aa3Rm9n3RBCq+5jgnTdKEv
'' SIG '' sFR2wZBPlOyGYf/bES+SAzDOMLeLD11Es0MdI1DNkdcv
'' SIG '' nfv8zbHBp8QOxO9APhk6AtQxqWmgSfl14ZvoaORqDI/r
'' SIG '' 5LEhe4ZnWH5/H+gr5BSyFtaBocraMJBr7m91wLA2JrII
'' SIG '' O/+9vn9sExjfxm2keUmti39hhwVo99Rw40KV6J67m0uy
'' SIG '' 4rZBPeevpxooya1hsKBBGBlO7UebYZXtPgthWuo+epiS
'' SIG '' Uc0/yUTngIspQnL3ebLdhOon7v59emsCAwEAAaOCAYsw
'' SIG '' ggGHMA4GA1UdDwEB/wQEAwIHgDAMBgNVHRMBAf8EAjAA
'' SIG '' MBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMCAGA1UdIAQZ
'' SIG '' MBcwCAYGZ4EMAQQCMAsGCWCGSAGG/WwHATAfBgNVHSME
'' SIG '' GDAWgBS6FtltTYUvcyl2mi91jGogj57IbzAdBgNVHQ4E
'' SIG '' FgQUjWS3iSH+VlhEhGGn6m8cNo/drw0wWgYDVR0fBFMw
'' SIG '' UTBPoE2gS4ZJaHR0cDovL2NybDMuZGlnaWNlcnQuY29t
'' SIG '' L0RpZ2lDZXJ0VHJ1c3RlZEc0UlNBNDA5NlNIQTI1NlRp
'' SIG '' bWVTdGFtcGluZ0NBLmNybDCBkAYIKwYBBQUHAQEEgYMw
'' SIG '' gYAwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2lj
'' SIG '' ZXJ0LmNvbTBYBggrBgEFBQcwAoZMaHR0cDovL2NhY2Vy
'' SIG '' dHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0
'' SIG '' UlNBNDA5NlNIQTI1NlRpbWVTdGFtcGluZ0NBLmNydDAN
'' SIG '' BgkqhkiG9w0BAQsFAAOCAgEADS0jdKbR9fjqS5k/AeT2
'' SIG '' DOSvFp3Zs4yXgimcQ28BLas4tXARv4QZiz9d5YZPvpM6
'' SIG '' 3io5WjlO2IRZpbwbmKrobO/RSGkZOFvPiTkdcHDZTt8j
'' SIG '' ImzV3/ZZy6HC6kx2yqHcoSuWuJtVqRprfdH1AglPgtal
'' SIG '' c4jEmIDf7kmVt7PMxafuDuHvHjiKn+8RyTFKWLbfOHzL
'' SIG '' +lz35FO/bgp8ftfemNUpZYkPopzAZfQBImXH6l50pls1
'' SIG '' klB89Bemh2RPPkaJFmMga8vye9A140pwSKm25x1gvQQi
'' SIG '' FSVwBnKpRDtpRxHT7unHoD5PELkwNuTzqmkJqIt+ZKJl
'' SIG '' lBH7bjLx9bs4rc3AkxHVMnhKSzcqTPNc3LaFwLtwMFV4
'' SIG '' 1pj+VG1/calIGnjdRncuG3rAM4r4SiiMEqhzzy350yPy
'' SIG '' nhngDZQooOvbGlGglYKOKGukzp123qlzqkhqWUOuX+r4
'' SIG '' DwZCnd8GaJb+KqB0W2Nm3mssuHiqTXBt8CzxBxV+NbTm
'' SIG '' tQyimaXXFWs1DoXW4CzM4AwkuHxSCx6ZfO/IyMWMWGmv
'' SIG '' qz3hz8x9Fa4Uv4px38qXsdhH6hyF4EVOEhwUKVjMb9N/
'' SIG '' y77BDkpvIJyu2XMyWQjnLZKhGhH+MpimXSuX4IvTnMxt
'' SIG '' tQ2uR2M4RxdbbxPaahBuH0m3RFu0CAqHWlkEdhGhp3cC
'' SIG '' ExwwggauMIIElqADAgECAhAHNje3JFR82Ees/ShmKl5b
'' SIG '' MA0GCSqGSIb3DQEBCwUAMGIxCzAJBgNVBAYTAlVTMRUw
'' SIG '' EwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3
'' SIG '' dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0
'' SIG '' IFRydXN0ZWQgUm9vdCBHNDAeFw0yMjAzMjMwMDAwMDBa
'' SIG '' Fw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVTMRcw
'' SIG '' FQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMy
'' SIG '' RGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNIQTI1
'' SIG '' NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqGSIb3DQEB
'' SIG '' AQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE
'' SIG '' 8pE3qZdRodbSg9GeTKJtoLDMg/la9hGhRBVCX6SI82j6
'' SIG '' ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9/UO0hNoR8XOx
'' SIG '' s+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt
'' SIG '' 69OxtXXnHwZljZQp09nsad/ZkIdGAHvbREGJ3HxqV3rw
'' SIG '' N3mfXazL6IRktFLydkf3YYMZ3V+0VAshaG43IbtArF+y
'' SIG '' 3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLF
'' SIG '' uk4fsbVYTXn+149zk6wsOeKlSNbwsDETqVcplicu9Yem
'' SIG '' j052FVUmcJgmf6AaRyBD40NjgHt1biclkJg6OBGz9vae
'' SIG '' 5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0
'' SIG '' QCirc0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/w
'' SIG '' ojzKQtwYSH8UNM/STKvvmz3+DrhkKvp1KCRB7UK/BZxm
'' SIG '' SVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T/jnA
'' SIG '' +bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjB
'' SIG '' Jgj5FBASA31fI7tk42PgpuE+9sJ0sj8eCXbsq11GdeJg
'' SIG '' o1gJASgADoRU7s7pXcheMBK9Rp6103a50g5rmQzSM7TN
'' SIG '' sQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIB
'' SIG '' ADAdBgNVHQ4EFgQUuhbZbU2FL3MpdpovdYxqII+eyG8w
'' SIG '' HwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08w
'' SIG '' DgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUF
'' SIG '' BwMIMHcGCCsGAQUFBwEBBGswaTAkBggrBgEFBQcwAYYY
'' SIG '' aHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUF
'' SIG '' BzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20v
'' SIG '' RGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8E
'' SIG '' PDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5j
'' SIG '' b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNV
'' SIG '' HSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJ
'' SIG '' KoZIhvcNAQELBQADggIBAH1ZjsCTtm+YqUQiAX5m1tgh
'' SIG '' QuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxpwc8d
'' SIG '' B+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiC
'' SIG '' qBa9qVbPFXONASIlzpVpP0d3+3J0FNf/q0+KLHqrhc1D
'' SIG '' X+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQcAp876i8
'' SIG '' dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6Z
'' SIG '' JxurJB4mwbfeKuv2nrF5mYGjVoarCkXJ38SNoOeY+/um
'' SIG '' nXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+jSbl3ZpHxcpzp
'' SIG '' SwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNB
'' SIG '' zU+2QJshIUDQtxMkzdwdeDrknq3lNHGS1yZr5Dhzq6YB
'' SIG '' T70/O3itTK37xJV77QpfMzmHQXh6OOmc4d0j/R0o08f5
'' SIG '' 6PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZ
'' SIG '' ynDwN7+YAN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3
'' SIG '' OtUVmDG0YgkPCr2B2RP+v6TR81fZvAT6gt4y3wSJ8ADN
'' SIG '' XcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2
'' SIG '' VVQrH4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8
'' SIG '' MYIDdjCCA3ICAQEwdzBjMQswCQYDVQQGEwJVUzEXMBUG
'' SIG '' A1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRp
'' SIG '' Z2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBTSEEyNTYg
'' SIG '' VGltZVN0YW1waW5nIENBAhAKekqInsmZQpAGYzhNhped
'' SIG '' MA0GCWCGSAFlAwQCAQUAoIHRMBoGCSqGSIb3DQEJAzEN
'' SIG '' BgsqhkiG9w0BCRABBDAcBgkqhkiG9w0BCQUxDxcNMjIw
'' SIG '' NDE4MTkwMzMzWjArBgsqhkiG9w0BCRACDDEcMBowGDAW
'' SIG '' BBSFCPOGUVyz0wd9trS3wH8bSl5B3jAvBgkqhkiG9w0B
'' SIG '' CQQxIgQgTBBVTP6ywBUf6fRipTT33+FJrddTXlAmVpEV
'' SIG '' rZs343cwNwYLKoZIhvcNAQkQAi8xKDAmMCQwIgQgnaaQ
'' SIG '' FcNJxsGJeEW6NYKtcMiPpCk722q+nCvSU5J55jswDQYJ
'' SIG '' KoZIhvcNAQEBBQAEggIANnTWuSaFk5kyWtSQnwe5LXo3
'' SIG '' CQ3512zf2ceZXx7MQ0dCEvDXrwpcxk4sqa1031VyZ3DM
'' SIG '' LGnDnwuffsMBO3gx0vqq0ay7v4E6qw1ShKK4U87er8ep
'' SIG '' liK/p/7/BXDAG6hlBreur5RIaPonv6/8Pglgs/QH/yYu
'' SIG '' 137ujIHhhGyXTWGhG8t8KiqnvNwrE58blBMr72pSOkBW
'' SIG '' CWOw0KozvGnJWm80kd7ITYoEDNZFT6nd5slLQyxfcSWh
'' SIG '' 9loIW8PjfTvzYAAkFdyEKnAogwFVUMHhJTHQgGTPBNn/
'' SIG '' DV8wN9XiKC0RJTl+PUSUyolMdaxfJgW/I3nc6pdayK8w
'' SIG '' CSCUOXg0TPjoAXYgAeUJBllpoEa+HCxKvc+mw5ECwOBO
'' SIG '' BygJbXkiPIHfOxMRKTxpJAzNuBix1OdUlvVXes15iZST
'' SIG '' 6G+Ed7AIPI/myi4yPGmvvGWXvO06CQ2+GjgJUdCUKVPx
'' SIG '' z1HNYvi4Y+vCtsf1jLjb5oApSUxjooGjGpMcRG43y/i2
'' SIG '' r4CSLZM/jbzbMPDMN10l125sj2hj3bRCjaK0DzDGGcb0
'' SIG '' XjOM9wrEoBZoU4j8VtuikLMXvF6slj//6R1AoiAxR/Im
'' SIG '' XlDjFZ/qdbdX7Vzrqzd4Sc1LB6/5gtn/mDsx37CoaiRp
'' SIG '' UV0QZ6y8QBK6PAklDIdM/9Ob2OXUERVrsLP3EsLGQ0o=
'' SIG '' End signature block
